/*******************************************************************************
* Copyright (c) 2000, 2004 IBM Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
*
* Contributors:
*     IBM Corporation - initial API and implementation
*******************************************************************************/
package org.eclipse.wst.rdb.internal.outputview;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Vector;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CTabFolder;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.custom.TableEditor;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.TabFolder;
import org.eclipse.swt.widgets.TabItem;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.help.WorkbenchHelp;
import org.eclipse.ui.part.ViewPart;
import org.w3c.dom.Document;

/**
 * A view part that presents an action table
 * and associated output tabbed panes for each action.
 */
public class OutputView extends ViewPart implements SelectionListener        
{
   /* Obsolete? */
   protected final static String PREFIX = "output_view.";
   /** Translated resource for the messages tab title. */
   protected final static String messages = OutputViewPlugin.getString("OV_DISPLAY_MESSAGES");
   /** Translated resource for the parameters tab title. */
   protected final static String parms = OutputViewPlugin.getString("OV_DISPLAY_PARMS");
   /** Translated resource for the results tab title. */
   protected final static String results = OutputViewPlugin.getString("OV_DISPLAY_RESULTS");
   /** Translated resource for the input column of the parameters tab. */
   protected final static String input = OutputViewPlugin.getString("OV_PARMS_INPUT");
   /** Translated resource for the output column of the parameters tab. */
   protected final static String output = OutputViewPlugin.getString("OV_PARMS_OUTPUT");
   /** Translated resource for the parameters tab title. */
   protected final static String name = OutputViewPlugin.getString("OV_PARMS_NAME");
   
   // begin - for processing the org.eclipse.wst.rdb.outputview.additionalOutputViewTabs extension point
   protected static final String ADDITIONAL_TABS_EXT_POINT = 
       "org.eclipse.wst.rdb.outputview.additionalOutputViewTabs";
   protected static final String ADDITIONAL_TAB_ATTR_CLASS = 
       "class";
   protected static final String ADDITIONAL_TAB_ATTR_ID = 
       "id";
  
   protected static ArrayList contributedTabOrder;
   protected static HashMap contributedTabs;
   // end   - for processing the org.eclipse.wst.rdb.outputview.additionalOutputViewTabs extension point
   
   protected Composite displayArea, arrowPane, topPane, resultArea;
   protected TabFolder pageArea;
   protected TabItem messageTab, parmsTab, resultsTab = null;
   protected OutputViewActionsTable actionsTable;
   protected ResultTableViewer resultViewer;
   protected final static OutputViewModel model = new OutputViewModel();
   protected TableViewer parmsTable, resultsTable;
   protected StyledText messageArea;
   protected Label nameLabel;
   protected Button leftArrowB, rightArrowB;
   protected OutputItem item;
   protected String msg;
   protected Vector parameters;
   protected static String BUTTON_TEXT = "...";
   protected ArrayList longParams = new ArrayList();
   protected static final int MAXDISPLAYLENGTH = 60;

   /**
    * Initializes the view.
    * @see ViewPart#init(IViewSite)
    */
   public void init(IViewSite site) throws PartInitException
   {
      super.init(site);
   }

   /**
    * Creates a part control, given a parent.
    * @param parent A parent composite.
    * @see ViewPart#createChild(IWorkbenchPartContainer)
    */
   public void createPartControl(Composite parent)
   {
      //Sash Form with tableViewer on left and 3 paned Display Area on right
      SashForm mainForm = new SashForm(parent, SWT.HORIZONTAL);
      mainForm.setLayoutData(new GridData(GridData.FILL_BOTH));

      //left side
      actionsTable = new OutputViewActionsTable(mainForm, model, this);
      actionsTable.getTable().addSelectionListener(this);
      //right side
      createDisplayArea(mainForm);

      mainForm.setWeights(new int[]{40, 60});

      WorkbenchHelp.setHelp(mainForm,"org.eclipse.wst.rdb.outputview.dboutput_view");
      
      refreshWithDefault();
   }

   /**
    * Creates a display area.
    * @param parent A parent composite.
    */
   protected void createDisplayArea(Composite parent)
   {
      //  Build the top portion of the display area, show the messageArea by default
      GridData gd;
      GridLayout gridLayout = new GridLayout();
      gridLayout.marginWidth = 0;
      gridLayout.marginHeight = 0;
      gridLayout.horizontalSpacing = 0;
      gridLayout.verticalSpacing = 0;
      //gridLayout.numColumns = 3;

      displayArea = new Composite(parent, SWT.NONE);
      displayArea.setLayout(gridLayout);

      topPane = new Composite(displayArea, SWT.NONE);
      FormLayout fl = new FormLayout();
      fl.marginHeight=2;
      fl.marginWidth = 0;
      topPane.setLayout(fl);

      // put label in composite so that label will no spill over arrow buttons
      Composite labelPane = new Composite(topPane, SWT.NONE);
      nameLabel = new Label(labelPane, SWT.NULL);

      nameLabel.setVisible(true);
      gd = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
      gd.grabExcessHorizontalSpace = true;
      nameLabel.setLayoutData(gd);

      arrowPane = createArrowPane(topPane);      // a composite

      // place arrow pane in topPane
      FormData fd = new FormData();
      fd.width = 50;
      fd.right = new FormAttachment(100,0);
      arrowPane.setLayoutData(fd);

      //place composite with label in top pane
      fd = new FormData();
      fd.right = new FormAttachment(arrowPane,0);
      fd.left = new FormAttachment(0,0);
      labelPane.setLayoutData(fd);

      gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
      topPane.setLayoutData(gd);

      // Build the set of 3 tabs for the messages, parameters and result sets
      pageArea = new TabFolder(displayArea, SWT.TOP);
      //	 pageArea.setTabHeight(16);
      pageArea.addSelectionListener(this);

      gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
      gd.verticalAlignment = GridData.FILL;
      gd.grabExcessVerticalSpace = true;
      gd.grabExcessHorizontalSpace = true;
      pageArea.setLayoutData(gd);

      //Page 1 - the Message Page
      messageTab	= new TabItem(pageArea, SWT.NULL);
      messageTab.setText(OutputViewPlugin.getString("OV_DISPLAY_MESSAGES"));
      messageTab.setToolTipText(OutputViewPlugin.getString("OV_MESSAGES_TOOL_TIP"));
      // The message area should stay left-to-right at all times.  
      // (That is, do not mirror for BiDi languages).
      messageArea = new StyledText(pageArea, SWT.V_SCROLL| SWT.MULTI | SWT.BORDER |
              SWT.READ_ONLY | SWT.WRAP | SWT.LEFT_TO_RIGHT);

      // menu for Message page
      OVMessageAreaMenu ovMessageMenu = new OVMessageAreaMenu(messageArea);
      messageArea.setMenu(ovMessageMenu.getMenu());

      //messageArea.setBackground(new Color(null,255, 255, 255));  //white
      messageTab.setControl(messageArea);

      //Page 2 - the Paramaters Page
      parmsTab	= new TabItem(pageArea, SWT.NULL);
      parmsTab.setText(OutputViewPlugin.getString("OV_DISPLAY_PARMS"));
      parmsTab.setToolTipText(OutputViewPlugin.getString("OV_PARMS_TOOL_TIP"));
      parmsTable = createParametersTable(pageArea);
      parmsTab.setControl(parmsTable.getControl());
      
      // menu for Parameters page
      OVParameterAreaMenu  ovParmMenu = new OVParameterAreaMenu(parmsTable.getTable());
      parmsTable.getTable().setMenu(ovParmMenu.getMenu());

      //Page 3 - the Result Set Page
      resultsTab	= new TabItem(pageArea, SWT.NULL);
      resultsTab.setText(OutputViewPlugin.getString("OV_DISPLAY_RESULTS"));
      resultsTab.setToolTipText(OutputViewPlugin.getString("OV_RESULTS_TOOL_TIP"));

      resultViewer = new ResultTableViewer(pageArea, model.getResultTableModel());      
      resultsTable = resultViewer.getTableViewer();
      resultsTab.setControl(resultsTable.getControl());
      
      // menu for Results page
      OVResultAreaMenu ovResultMenu = new OVResultAreaMenu(resultsTable.getTable());
      resultsTable.getTable().setMenu(ovResultMenu.getMenu());

      // add in additional tabs that are contributed via the additionalOutputViewTabs extension point
      addTabItemsFromExtensions ();
      
      //by default select message page
      showMessagePage();
   }

   /*
    * addTabItemsFromExtensions
    * Read in all contributions to the org.eclipse.wst.rdb.outputview.additionalOutputViewTabs extension point
    */
   private void addTabItemsFromExtensions() {
	   IExtensionRegistry extensionRegistry = Platform.getExtensionRegistry();
	   IExtensionPoint extensionPoint = 
		   extensionRegistry.getExtensionPoint(ADDITIONAL_TABS_EXT_POINT);
	   IExtension [] extensions = extensionPoint.getExtensions(); 
	   contributedTabs = new LinkedHashMap (extensions.length);
	   contributedTabOrder = new ArrayList (extensions.length);
	   // read in all additionalTab elements that contribute to the
	   // org.eclipse.wst.rdb.outputview.additionalOutputViewTabs 
	   for (int i=0;i<extensions.length;i++)
	   {
		   IExtension ext = extensions[i];
		   IConfigurationElement [] configElements = ext.getConfigurationElements();
		   for (int j=0;j<configElements.length;j++)
		   {
			   IConfigurationElement currentConfigElement = configElements[j];
			   try
			   {                    
				   IAdditionalOutputViewTab myTabs = 
					   (IAdditionalOutputViewTab)currentConfigElement.createExecutableExtension(ADDITIONAL_TAB_ATTR_CLASS);
				   String id = currentConfigElement.getAttribute (ADDITIONAL_TAB_ATTR_ID);
				   // set the tab in the pageArea (TabFolder)
				   myTabs.getTab(pageArea);
				   contributedTabs.put(id, myTabs);
				   contributedTabOrder.add(id);
			   }
			   catch (CoreException ex)
			   {
				   OutputViewPlugin.getPlugin().writeLog(IStatus.ERROR, 0, 
						   "OutputView:addTabItemsFromExtensions()", ex); //$NON-NLS-1$
			   }
		   }
	   }	   
   }

   /**
    * Creates a parameters table.
    * @param parent A parent composite.
    */
   protected TableViewer createParametersTable(Composite parent) {

      Table aTable = new Table(parent,SWT.SINGLE | SWT.FULL_SELECTION | SWT.BORDER | SWT.H_SCROLL | SWT.V_SCROLL);
      aTable.setLinesVisible (true);
      aTable.setHeaderVisible (true);
      aTable.addSelectionListener(this);

      TableColumn column = new TableColumn(aTable, SWT.NULL);
      column.setText(name);
      column = new TableColumn(aTable, SWT.NULL);
      column.setText(input);
      column = new TableColumn(aTable, SWT.NULL);
      column.setText(output);

      parmsTable = new TableViewer(aTable);

      TableLayout layout = new TableLayout();
      ColumnWeightData data = new ColumnWeightData(33,true);
      layout.addColumnData(data);
      layout.addColumnData(data);
      layout.addColumnData(new ColumnWeightData(34, false));
      aTable.setLayout(layout);

      parmsTable.setLabelProvider(new ParmsTableLabelProvider());
      parmsTable.setContentProvider(new ParmsTableContentProvider());

      parameters = new Vector();
      parmsTable.setInput(parameters);
      return parmsTable;
   }

   /**
    * Creates an arrow pane.
    * @param parent A parent composite.
    */
   protected Composite createArrowPane(Composite parent)
   {
      arrowPane = new Composite(parent, SWT.NONE);
      GridLayout layout = new GridLayout();
      layout.marginWidth = 0;
      layout.marginHeight = 0;
      layout.numColumns = 2;
      layout.horizontalSpacing = 0;
      layout.verticalSpacing = 0;
      arrowPane.setLayout(layout);

      leftArrowB = new Button(arrowPane, SWT.PUSH);
      leftArrowB.setImage( OutputViewPlugin.getPlugin().
               getImageDescriptor("left_arrow_nav").createImage());
      leftArrowB.setToolTipText(OutputViewPlugin.getString("OV_RESULTS_LEFTARROW"));
      leftArrowB.addSelectionListener(this);
      GridData gd = new GridData();
      gd.widthHint = 24;
      gd.heightHint = 18;
      gd.verticalAlignment = GridData.BEGINNING;
      gd.horizontalAlignment = GridData.FILL;
      leftArrowB.setLayoutData(gd);

      rightArrowB = new Button(arrowPane, SWT.PUSH);
      rightArrowB.setImage( OutputViewPlugin.getPlugin().
               getImageDescriptor("right_arrow_nav").createImage());
      rightArrowB.setToolTipText(OutputViewPlugin.getString("OV_RESULTS_RIGHTARROW"));
      rightArrowB.addSelectionListener(this);
      gd = new GridData();
      gd.widthHint = 24;
      gd.heightHint = 18;
      gd.verticalAlignment = GridData.BEGINNING;
      gd.horizontalAlignment = GridData.FILL;
      rightArrowB.setLayoutData(gd);

      return arrowPane;
   }

   /**
    * Gets the model.
    * @return The OutputViewModel.
    */
   protected OutputViewModel getModel()
   {
      return model;
   }

   /**
    * Adds an output item.
    * @param The item to add.
    *     @param async       true if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        false, if the update is to be performed synchronously.
    * @param showMsgPage True if we should show the messages tab.
    */
   protected void addOutputItem(final OutputItem item, final boolean async, final boolean showMsgPage)
   {
      model.addOutputItem(item);
      if (async)
      {
         // If view part has not been initialized yet,
         // ignore adding the item - since it can't be done.	
         if (displayArea == null)	
            return;

         Display display = displayArea.getDisplay();
         if (display != null)
         {

            display.asyncExec(new Runnable() {
               public void run(){

                  actionsTable.refresh();
                  updateContentForNewSelectedItem(item);
                  int i = model.getActionList().indexOf(item);
                  actionsTable.selectItem(i);
                  if (showMsgPage){
                     showMessagePage(false);
                  }
                  else{
                     if (item.hasResults() && item.isStatusSuccess())
                        showResultsPage(false);
                     else
                        showMessagePage(false);
                  }
               }
            });
         }  // display != null
      }  // async
      else
      {
         actionsTable.refresh();
         updateContentForNewSelectedItem(item);
         int i = model.getActionList().indexOf(item);
         actionsTable.selectItem(i);
         if (showMsgPage)
            showMessagePage(false);
         else
         {
            if (item.hasResults() && item.isStatusSuccess())
               showResultsPage(false);
            else
               showMessagePage(false);
         }
      }
   }

   /**
    * Appends a message to the messages tab.
    * @param item The item.
    * @param msg The message.
    * @param async       True if the update is to be performed asynchronously, meaning
    *                    the update is delayed to the event dispath thread via
    *                    the Display syncExec method.
    *                    False if the update is to be performed synchronously.
    */
   protected void showMessage(OutputItem item, String msg, boolean async)
   {
      item.appendMessage(msg);
      addOutputItem(item, async, true);
   }

   /**
    * Adds an output item.
    * @param item The item.
    * @param async       True if the update is to be performed asynchronously, meaning
    *                    the update is delayed to the event dispath thread via
    *                    the Display syncExec method.
    *                    False if the update is to be performed synchronously.
    */
   protected void showMessage(OutputItem item, boolean async)
   {
      addOutputItem(item, async, true);
   }

   /**
    * Shows the result tab for a given item.
    * @param item The item.
    */
   public void showResult(final OutputItem item)
   {
      Display display = displayArea.getDisplay();
      if (display != null)
      {
         display.asyncExec(new Runnable() {
            public void run(){
               // an item is added to the table if it is a new OututItem, need to refresh the table
               actionsTable.refresh();
               updateContentForNewSelectedItem(item);
               showResultsPage();
               actionsTable.selectItem(item);
            }
         }); // new Runnable()
      }
   }

   /**
    * Shows the result tab for a given item.
    * @param item The item.
    * @param async       True if the update is to be performed asynchronously, meaning
    *                    the update is delayed to the event dispath thread via
    *                    the Display syncExec method.
    *                    False if the update is to be performed synchronously.
    */
   public void showResult(OutputItem item, boolean async)
   {
      if (async){
         showResult(item,true);
      }
      else{
         actionsTable.refresh();
         updateContentForNewSelectedItem(item);
         showResultsPage();
         actionsTable.selectItem(item);
      }  		

   }

   /**
    * Shows the parameters tab for a given item.
    * @param item The item.
    * @param async       True if the update is to be performed asynchronously, meaning
    *                    the update is delayed to the event dispath thread via
    *                    the Display syncExec method.
    *                    False if the update is to be performed synchronously.
    */
   public void showParameters(OutputItem item, boolean async)
   {
      if (async){
         showParameters(item,null);
      }
      else{
         actionsTable.refresh();
         updateContentForNewSelectedItem(item);
         showParametersPage();
         actionsTable.selectItem(item);
      }

   }

   /**
    * Displays the parameters for a given item.
    * @param item The item.
    * @param parms The parameters.
    */
   public void showParameters(OutputItem item, Vector parms)
   {
      // if null, use what is already in output view model
      if (parms != null)
         model.addParameters(item, parms);
      Display display = displayArea.getDisplay();
      if (display != null)
      {
         final OutputItem outputItem = item;
         display.asyncExec(new Runnable() {
            public void run(){
               actionsTable.refresh();
               updateContentForNewSelectedItem(outputItem);
               showParametersPage();
               actionsTable.selectItem(outputItem);
            }
         }); // new Runnable()
      }
   }

   /**
    * Updates the messages tab from the messages stored in the model.
    * @param item The item.
    */
   protected void updateMessage(OutputItem item)
   {
      nameLabel.setText(item.getUniqueName());
      nameLabel.setToolTipText(item.getUniqueName());
      if (item.getMessages() != null) {
         messageArea.setText(item.getMessages());
         messageArea.setTopIndex(messageArea.getLineCount());
      }
      nameLabel.pack();
   }

   /**
    * Updates the results tab from the results stored in the model.
    * @param item The item.
    */
   protected void updateResults(OutputItem item)
   {

      if (item.hasResults())
      {
         ResultTableModel tabModel =  model.getResultTableModel(item);
         resultViewer.setTableModel(tabModel);
         resultViewer.refresh();
      }
      else
      {
         clearResults();
         //resultsTable.getTable().setVisible(false);
      }
      updateArrows(item);
   }

   /**
    * Updates the parameters tab from the parameters stored in the model.
    * @param item The item.
    */
   protected void updateParameters(OutputItem item)
   {
      updateParameters(item, model.getParameters(item));
   }

   /**
    * Updates the parameters tab.
    * @param item The item.
    * @param params The parameters.
    */
   protected void updateParameters(OutputItem item, Vector parms)
   {
      if (item.hasParameters()  && parms != null && !parms.isEmpty())
      {
         if (parameters.isEmpty() || parms.get(0) != parameters.get(0))	{
            clearParameters();
            parameters.addAll(parms);

            parmsTable.refresh();
            // Add buttons to table if param length > MAXDISPLAYLENGTH
            addParamButtons();
         }
      }
      else
         clearParameters();
   }

   /**
    * Clear all additionalOutputViewTabs for the specified OutputItem
    * @param item the OutputItem to clear out all additionalOutputViewTabs for 
    */
   protected void clearAllAdditionalTabs(OutputItem item) {
	   // for each contribution to the org.eclipse.wst.rdb.outputview.additionalOutputViewTabs extension point, update the contents
	   Iterator tabIDIterator = contributedTabs.keySet().iterator();
	   
	   while (tabIDIterator.hasNext()) {
		   String id = (String) tabIDIterator.next();
		   clearAdditionalTab(id);
	   }
   }
   
   protected void clearAdditionalTab(String tabID) {
	   if (tabID != null) {
		   IAdditionalOutputViewTab tab = (IAdditionalOutputViewTab)contributedTabs.get(tabID);
		   
		   if (tab != null) {
			   tab.clearContents();
			   tab.refresh();	   
		   }
	   }
   }
   
   /**
    * Update the contents for all contributed additional tabs
    * @param item OutputItem to update the contributed tabs for
    */   
   protected void updateAdditionalTabs(OutputItem item) {
	   // get all of the contributed tabs and check for data
	   for (Iterator itr = contributedTabs.keySet().iterator(); itr.hasNext(); /* no-op */) {
		   String tabID = (String)itr.next();
		   clearAdditionalTab(tabID);
		   
		   if (item.hasContents(tabID)) {			   
			   IAdditionalOutputViewTab tab = (IAdditionalOutputViewTab)contributedTabs.get(tabID);
			   if (tab != null) {
				   tab.setContents (getModel().getContents(item, tabID));
				   tab.refresh();	
			   }
		   }
	   }
   }   
   
   /**
    * Updates the messages, parameters, and results
    * from the information stored in the model.
    * @param item The item.
    */
   protected void updateContentForNewSelectedItem(OutputItem item)
   {
      updateMessage(item);
      updateParameters(item);
      updateResults(item);
      updateAdditionalTabs(item);      
   }

   /**
    * Replaces the messages for an item synchronously.
    * @param item The item.
    * @param msgs New messages.
    */
   protected void replaceMessage(OutputItem item, String msg)
   {
      replaceMessage(item, msg, false);
   }

   /**
    * Replaces the messages for an item
    * @param item The item.
    * @param msgs New messages.
    *     @param async       true if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        false, if the update is to be performed synchronously.
    */
   protected void replaceMessage(OutputItem item1, String msg1, boolean async)
   {
      this.item = item1;
      this.msg = msg1;
      if (model.outputItemExists(item))
      {
         if (async) {
            if (displayArea == null)
               return;
            Display display = displayArea.getDisplay();
            if (display != null) {
               display.asyncExec(new Runnable() {
                  public void run() {
                     item.setMessage(msg);
                     updateMessage(item);
                     actionsTable.selectItem(item);
                  }
               });
            }
         }else {
            item.setMessage(msg);
            updateMessage(item);
            actionsTable.selectItem(item);
         }
      }
   }

   /**
    * Clears the messages for an item.
    * @param item The item.
    */
   protected void clearMessages(OutputItem item)
   {
      if (model.outputItemExists(item))
      {
         item.setMessage("");
         updateMessage(item);
         actionsTable.selectItem(item);
      }
   }

   /**
    * Clears the messages, parameters, and results for the selected item.
    */
   protected void clearOutputPages()
   {
      messageArea.setText("");
      nameLabel.setText("");
      nameLabel.setToolTipText("");
      nameLabel.pack();
      clearParameters();
      clearResults();
   }

   /**
    * Clears the parameters for the selected item.
    */
   protected void clearParameters()
   {
      Iterator iter;
      parameters.clear();
      // Remove column controls for long parameters	
      if (longParams != null) {
         Iterator iterLongParams = longParams.iterator();
         while (iterLongParams.hasNext()) {
            LongParamControls longParamCtls =
               (LongParamControls) iterLongParams.next();

            iter = longParamCtls.getEditorCells().iterator();
            while (iter.hasNext()) {
               ((TableEditor) iter.next()).dispose();
            }

            iter = longParamCtls.getEditorButtons().iterator();
            while (iter.hasNext()) {
               ((Button) iter.next()).dispose();
            }
         }
         longParams.clear();
      }
      parmsTable.refresh();
   }

   /**
    * Clears the results for the selected item.
    */
   protected void clearResults()
   {
      resultViewer.clearResult();
      resultViewer.setTableModel(null);
      resultViewer.refresh();
   }

   /**
    * Clears the parameters for the selected item.
    */
   protected void clearParameters(OutputItem item)
   {
      actionsTable.selectItem(item);
      clearParameters();
   }

   /**
    * Clears the results for a given item.
    * @param item The item.
    */
   protected void clearResults(OutputItem item)
   {
      actionsTable.selectItem(item);
      resultViewer.clearResult();
      resultViewer.setTableModel(null);
      resultViewer.refresh();
   }

   /**
    * Refreshes the action table for a given item synchronously.
    * @param item The item.
    */
   protected void refreshActionTable(OutputItem item)
   {
      refreshActionTable(item, false);
   }

   /**
    * Refreshes the action table for a given item.
    * @param item1 The item.
    *     @param async       True if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        False if the update is to be performed synchronously.
    */
   protected void refreshActionTable(OutputItem item1, boolean async)
   {
      this.item = item1;
      if (async) {
         if (displayArea == null)
            return;
         Display display = displayArea.getDisplay();
         if (display != null) {
            display.asyncExec(new Runnable() {
               public void run() {
                  actionsTable.selectItem(item);
               }});
         }
      }else {
         actionsTable.selectItem(item);
      }
   }

   /**
    * Updates the arrows that navigate between multiple result sets
    * for a given item.
    * @param item The item.
    */
   protected void updateArrows(OutputItem item)
   {
      if (item != null && item.hasResults() && isResultsTabSelected())
      {        
        if ( model.isLastResultSet(item) )
            rightArrowB.setEnabled(false);
         else
            rightArrowB.setEnabled(true);

         if ( model.isFirstResultSet(item) )
            leftArrowB.setEnabled(false);
         else
            leftArrowB.setEnabled(true);
         arrowPane.pack();
      }
      else
         disableArrows();

   }

   /**
    * Disables the arrows.
    */
   protected void disableArrows()
   {
      leftArrowB.setEnabled(false);
      rightArrowB.setEnabled(false);
      arrowPane.pack();
   }

   /**
    * Refreshes the output and reselects the first item, if there is one.
    */
   protected void refreshWithDefault()
   {
      if (model.isActionListEmpty())
      {
         actionsTable.refresh();
         clearOutputPages();
      }
      else
      {
         actionsTable.selectItem(0);
         updateContentForNewSelectedItem( model.getItem(0) );
         showMessagePage();
      }
   }

   /**
    * Shows the messages synchronously.
    */
   protected void showMessagePage(){
      showMessagePage(false);
   }

   /**'
    * Shows the messages.
    *     @param async       true if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        false, if the update is to be performed synchronously.
    */
   protected void showMessagePage(boolean async)
   {
      if (async) {
         if (displayArea == null)
            return;
         Display display = displayArea.getDisplay();
         if (display != null) {
            display.asyncExec(new Runnable() {
               public void run() {
                  pageArea.setSelection(0);
                  disableArrows();
               }});
         }
      }else {
         pageArea.setSelection(0);
         disableArrows();
      }
   }

   /**
    * Shows the parameters synchronously.
    */
   protected void showParametersPage(){
      showParametersPage(false);
   }

   /**
    * Shows the parameters.
    *     @param async       true if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        false, if the update is to be performed synchronously.
    */
   protected void showParametersPage(boolean async)
   {
      if (async) {
         if (displayArea == null)
            return;
         Display display = displayArea.getDisplay();
         if (display != null) {
            display.asyncExec(new Runnable() {
               public void run() {
                  pageArea.setSelection(1);
                  disableArrows();
               }});
         }
      }else {
         pageArea.setSelection(1);
         disableArrows();
      }
   }

   /**
    * Shows the results synchronously.
    */
   protected void showResultsPage() {
      showResultsPage(false);
      OutputItem item = actionsTable.getSelectedItem();
      updateArrows(item);
   }

   /**
    * Shows the results.
    *     @param async       true if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        false, if the update is to be performed synchronously.
    */
   protected void showResultsPage(boolean async)
   {
      if (async) {
         if (displayArea == null)
            return;
         Display display = displayArea.getDisplay();
         if (display != null) {
            display.asyncExec(new Runnable() {
               public void run() {
                  pageArea.setSelection(2);
                  //arrowPane.setVisible(true);
               }});
         }
      }else {
         pageArea.setSelection(2);
      }
   }


   /**
    * Handles selection events.
    * @param e The selection event.
    */
   public void widgetSelected(SelectionEvent e)
   {
      Object source = e.getSource();
      try{	
         if (source.equals(actionsTable.getTable()) )
         {
            OutputItem item =  actionsTable.getSelectedItem();
            displayArea.setVisible(false);  // avoid flickering                  
            updateContentForNewSelectedItem(item); 
            showMessagePage();            
            displayArea.setVisible(true);
         }
         else if (source.equals(leftArrowB) )
         {
            OutputItem item = actionsTable.getSelectedItem();
            if (item != null)
            {
               model.getPreviousResultSetModel(item);
               displayArea.setVisible(false);  // avoid flickering
               updateResults(item);
               updateArrows(item);
               displayArea.setVisible(true);
            }
         }
         else if (source.equals(rightArrowB) )
         {
            OutputItem item = actionsTable.getSelectedItem();
            if (item != null)
            {
               model.getNextResultSetModel(item);
               displayArea.setVisible(false);  // avoid flickering
               updateResults(item);
               updateArrows(item);
               displayArea.setVisible(true);
            }
         }
         else if (source == pageArea)
         {
            // Disable left and right arrows except for results tab
            OutputItem item = actionsTable.getSelectedItem();
            updateArrows(item);                   
         }
         else if (source instanceof CTabFolder )
         {
            if ( ((CTabFolder)source).getSelectionIndex() == 2 )
            {
               updateArrows(actionsTable.getSelectedItem());
            }
         }
      }
      catch (SQLException ex)
      {
        OutputViewPlugin.getPlugin().writeLog(IStatus.ERROR, 0, 
                "OutputView:widgetSelected()", ex); //$NON-NLS-1$    	
      }
   }


   /**
    * Delegages selection events to widgetSelected.
    * @param e The selection event.
    */
   public void widgetDefaultSelected(SelectionEvent e)
   {
      widgetSelected(e);
   }
   
   /**
    * Determines whether or not the selected tab item is the results tab
    * @return true if selected item is results tab, false if not
    */
   private boolean isResultsTabSelected()
   {
   	    int index = pageArea.getSelectionIndex();
        if (index > -1)
        {
        	TabItem selectedItem = pageArea.getItem(index);
            if (selectedItem == resultsTab)
            {
            	return true;
            }
        }
        return false;
   }   

   /**
    * A content provider for the content of the parmeters table in the form
    * of an array of parameters.
    */
   protected class ParmsTableContentProvider implements IStructuredContentProvider
   {
      /**
       * Gets the elements of a Vector as an array.
       * @param arg0 An object of type Vector.
       * @return An array of objects (one for each row), or null if the input is not a Vector.
       *
       * @see IStructuredContentProvider#getElements(Object)
       */
      public Object[] getElements(Object collection) {

         if (collection instanceof Vector)
         {
            return ((Vector)collection).toArray();
         }
         else
            return null;
      }

      /**
       * Does nothing.
       * @see IContentProvider#dispose()
       */
      public void dispose() {
      }

      /**
       * Does nothing.
       * @see IContentProvider#inputChanged(Viewer, Object, Object)
       */
      public void inputChanged(Viewer viewer, Object old, Object newobj)
      {
      }

   } //END ContentProvider

   /**
    * LabelProvider for the ActionsTable.
    */
   protected class ParmsTableLabelProvider extends LabelProvider implements ITableLabelProvider
   {
      /**
       * Gets the text for a given column and row.
       * @param object of type String[], representing a row
       * @param columnIndex The index of the column.
       * @return The result to be displayed in the cell.
       *
       * @see ITableLabelProvider#getColumnText(Object, int)
       */
      public String getColumnText(Object parmRow, int columnIndex) {

         // a row is an array of Strings
         if (parmRow instanceof String[])
         {
            String[] row = (String[])parmRow;
            if (row[columnIndex] == null)
               return "";
            else
               return (String)row[columnIndex];
         }
         else
            return "";
      }

      /**
       * Does nothing.
       * @return null.
       * @see ITableLabelProvider#getColumnImage(Object, int)
       */
      public Image getColumnImage(Object outputItem, int columnIndex) {
         return null;
      }

   } //END LabelProvider

   /**
    * Sets the focus on the actions table.
    */
   public void setFocus()
   {
      if (actionsTable != null)
         actionsTable.getTable().setFocus();
   }

   /**
    * Disposes the objects owned by this view.
    */
   public void dispose()
   {
      parameters = null;
      resultViewer = null;
      actionsTable = null;
      //LogOutput.stop();
      super.dispose();
   }

   /**
    * Adds controls to expand long parameter values in the parameters table.
    */
   protected void addParamButtons()
   {
      LongParamControls longParamCtls;
      // Add control for input parameters
      longParamCtls = new LongParamControls(1);
      longParams.add(longParamCtls);
      // Add control for output paramters
      longParamCtls = new LongParamControls(2);
      longParams.add(longParamCtls);		

      for (int i = 0; i < parameters.size(); i++)
      {
         // Dont add button for Binary columns - since data length is usually
         // too long for the dialog to dispay - and the IDE hangs
         String[] param = (String[]) parameters.get(i);
         TableItem[] items = parmsTable.getTable().getItems();	
         if (param[1] != null && param[1].length() > MAXDISPLAYLENGTH) {

            setParamButton(
                     i,
                     items,
                     (LongParamControls) longParams.get(0));
         }
         if (param[2] != null && param[2].length() > MAXDISPLAYLENGTH){

            setParamButton(
                     i,
                     items,
                     (LongParamControls) longParams.get(1));
         }
      }
      parmsTable.getTable().layout(true);

      // Workaround for LINUX redrawing problems
      redrawTable();

   }

   /**
    * Sets parameter buttons for a given parameter.
    * @param i Index of the table row.
    * @param items Objects for the cells in the row.
    * @param longParamCtls Manager of the ellipsis buttons.
    */
   protected void setParamButton(
            int i,              // table row index
            TableItem[] items,  // table of ResultTableRow(s)
            LongParamControls longParamCtls) {

      TableItem item = items[i];
      TableEditor tableEditor;
      Button colButton;

      Table table = parmsTable.getTable();
      tableEditor = new TableEditor(table);
      colButton = new Button(table, SWT.PUSH);

      colButton.setToolTipText(
               OutputViewPlugin.getString("OV_LOB_COLUMN_TOOLTIP"));
      colButton.setText(BUTTON_TEXT); // + butNo++);
      colButton.setSize(colButton.computeSize(SWT.DEFAULT, SWT.DEFAULT));
      colButton.setData( new Integer( i ) );  // for use in the ButtonListener

      tableEditor.horizontalAlignment = SWT.RIGHT;
      tableEditor.minimumWidth =
         colButton.computeSize(SWT.DEFAULT, SWT.DEFAULT).x;

      tableEditor.setEditor(colButton, item, longParamCtls.getColumnIndex());

      longParamCtls.getEditorCells().add(tableEditor);
      longParamCtls.getEditorButtons().add(colButton);

      colButton.addSelectionListener(longParamCtls.getListener());

   }

   /**
    * Redraws the parameters table.
    */
   protected void redrawTable()
   {
      // Fix for LINUX redrawing problem
      parmsTable.getTable().layout();
      parmsTable.getTable().getParent().layout();
      parmsTable.getTable().setSize(parmsTable.getTable().getSize().x+1, parmsTable.getTable().getSize().y);
      parmsTable.getTable().setSize(parmsTable.getTable().getSize().x-1, parmsTable.getTable().getSize().y);
   }

   /**
    * Show the specified contributed contents for the given OutputItem
    * @param item tem   The OutputItem for which to show the content
    * @param id The id attribute of the additionalTab element for the contribution to org.eclipse.wst.rdb.outputview.additionalOutputViewTabs, used to identify the proper contribution
    * @param async       True if the update is to be performed asynchronously, meaning
    *                        the update is delayed to the event dispath thread via
    *                        the Display syncExec method.
    *                        False if the update is to be performed synchronously.
    */
   public void showContents(OutputItem item, String id, boolean async)
   {
     if (async){
       showContents(item, id, null);
     }
     else{
       actionsTable.refresh();
       updateContentForNewSelectedItem(item);
       showContentsPage(id);
       actionsTable.selectItem(item);
     }
   }   
   
   public void showContents(OutputItem item, final String id, Object contents)
   {
	   // if null, use what is already in output view model
	   if (contents != null)
		   getModel().addContents(item, id, contents);
	   Display display = displayArea.getDisplay();
	   if (display != null)
	   {
		   final OutputItem outputItem = item;
		   display.asyncExec(new Runnable() {
			   public void run(){
				   actionsTable.refresh();
				   updateContentForNewSelectedItem(outputItem);
				   showContentsPage(id);
				   actionsTable.selectItem(outputItem);
			   }
		   }); // new Runnable()
	   }
   }
   
   protected void showContentsPage(String id){
	   showContentsPage(id, false);
   }
   
   protected void showContentsPage(final String id, boolean async)
   {
	   if (async) {
		   if (displayArea == null)
			   return;
		   Display display = displayArea.getDisplay();
		   if (display != null) {
			   display.asyncExec(new Runnable() {
				   public void run() {
					   pageArea.setSelection(getPageIndex(id));
				   }});
		   }
	   }else {
		   pageArea.setSelection(getPageIndex(id));
	   }
   }
   
   protected int getPageIndex (String id) {
	   int index = -1;
	   int i = 1;
	   Iterator tabIDIterator = contributedTabOrder.iterator();
	   while (tabIDIterator.hasNext()) {
		   String tabID = (String) tabIDIterator.next();
		   if (id.equals(tabID)) {
			   // add 2, since we have 3 standard tabs (messages, results, parameters).  Indexing starts at 0
			   index = 2 + i; 
		       break; 
		   }
		   i++;
	   }
	   return index;
   }
   
   /**
    * Listens for selection events on ellipsis buttons.
    */
   protected class ButtonListener extends SelectionAdapter
   {

      /**
       * Responds when the ellipsis buttons are selected.
       * @param e The event.
       * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
       */
      public void widgetSelected(SelectionEvent e) {
         LongParamControls longParamCtls = null;
         // when the button is clicked, get the text for this column and display it in
         //   a read only dialog.

         // Determine which column this listener is for
         Iterator iterLongParams = longParams.iterator();
         while (iterLongParams.hasNext()) {
            longParamCtls = (LongParamControls) iterLongParams.next();
            if (longParamCtls.getListener() == this)
               break;
         }
         Button button = (Button)e.widget;

         int i = ((Integer)button.getData()).intValue();  // when the button was constructed, the row in which
         // it is placed is cached here.
         String[] param = (String[]) parameters.get(i);

         String title;	
         if (longParamCtls.getColumnIndex() == 1)
            title = param[0] + " - " + input;
         else	
            title = param[0] + " - " + output;
         boolean truncated;
         if (param[3].equals("1"))
            truncated = true;
         else
            truncated = false;	
         
         // XML support
         Document doc = resultViewer.parseXMLText(param[longParamCtls.getColumnIndex()]);
         if (doc != null)
         {
         	// Launch XML view since it is a valid XML document
            DialogShowXMLField xmlDialog = new DialogShowXMLField(pageArea.getShell(),
                    title, param[longParamCtls.getColumnIndex()]);
            xmlDialog.open();
         }
         else
         {
         	DialogShowLongField ddlDialog =
         		new DialogShowLongField(
                     pageArea.getShell(),
                     title,
                     param[longParamCtls.getColumnIndex()], truncated);


         	ddlDialog.open();	
         }
      }
   }

   /**
    * Manages controls for expanding large values in table cells.
    */
   protected class LongParamControls
   {
      protected ArrayList editorCells;
      protected ArrayList editorButtons;
      protected ButtonListener listener;
      protected int colIndex;
      /**
       * Constructs the manager for a given column.
       * @param aColIndex The index of the column.
       */
      public LongParamControls (int aColIndex) {
         colIndex = aColIndex;
         editorCells = new ArrayList();
         editorButtons = new ArrayList();
         listener = new ButtonListener();
      }
      /**
       * Gets the editor cells.
       * @return An ArrayList of cells.
       */
      public ArrayList getEditorCells() {
         return editorCells;
      }
      /**
       * Gets the editor buttons.
       * @return An ArrayList of buttons.
       */
      public ArrayList getEditorButtons() {
         return editorButtons;
      }
      /**
       * Gets the Listener.
       * @return A listener that can listen to the editor buttons.
       */
      public ButtonListener getListener() {
         return listener;
      }
      /**
       * Gets the column this manager is managing.
       * @return The column index.
       */
      public int getColumnIndex() {
         return colIndex;
      }
   }

}